{*********************************************************}
{                                                         }
{    Calmira System Library 3.1                           }
{    by Li-Hsin Huang & Erwin Dokter                      }
{    released into the public domain january 2001         }
{                                                         }
{*********************************************************}

unit Profile;

{ TProfile is a simple extension of TIniFile that can read and
  write string lists, and properties of fonts and headers.
  It also contains 'missing' methods ported from D2 IniFiles.

  The strings and headers are stored in the way Delphi stores
  INI lists, e.g.

  [Strings]
  Count=2
  S0=abc
  S1=def

  Fonts are written so that the values can easily be edited, e.g.

  [Main window]
  FontName=MS Sans Serif
  FontSize=8
  FontStyle=Bold Italic }

interface

uses IniFiles, Classes, Graphics, ExtCtrls;

type
  TProfile = class(TIniFile)
  public
    { Public declarations }
    procedure DeleteKey(const Section, Ident: string); { 3.0 }
    procedure ReadSections(Strings: TStrings); { 3.1 }
    procedure ReadStrings(const Section: string; S: TStrings);
    procedure ReadNewStrings(const Section: string; S: TStrings);
    procedure WriteStrings(const Section: string; S: TStrings);
    procedure WriteNewStrings(const Section: string; S: TStrings);
    procedure WriteSectionValues(const Section: string; S: TStrings);
    procedure ReadFont(const Section: string; Font: TFont);
    procedure WriteFont(const Section: string; Font: TFont);
    procedure ReadHeader(const Section: string; Header: THeader);
    procedure WriteHeader(const Section: string; Header: THeader);
  end;

implementation

uses SysUtils, WinProcs, Strings;

procedure TProfile.DeleteKey(const Section, Ident: string); { 3.0 }
var
  CSection: array[0..127] of Char;
  CIdent: array[0..127] of Char;
  CFileName: array[0..79] of Char;
begin
  WritePrivateProfileString(StrPLCopy(CSection, Section, SizeOf(CSection) - 1),
    StrPLCopy(CIdent, Ident, SizeOf(CIdent) - 1), nil,
    StrPLCopy(CFileName, FileName, SizeOf(CFileName) - 1));
end;

procedure TProfile.ReadSections(Strings: TStrings); { 3.1 }
{ The method Delphi uses under Win32 (section = nil) to get
  all the section names, does not work under 16-bit Windows.
  We have to parse the file ourselves. }
var
  F: TextFile;
  S: string;
begin
  AssignFile(F, FileName);
  try
    Reset(F);
  except
    Exit;
  end;
  while not Eof(F) do
  begin
    Readln(F, S);
    if (S > '') and (S[1] = '[') then
      Strings.Add(Copy(S, 2, Length(S) - 2));
  end;
  CloseFile(F);
end;

procedure TProfile.ReadStrings(const Section: string; S: TStrings);
var
  i: Integer;
begin
  for i := 0 to ReadInteger(Section, 'Count', 0) - 1 do
    S.Add(ReadString(section, 'S' + IntToStr(i), ''));
end;

procedure TProfile.ReadNewStrings(const Section: string; S: TStrings);
begin
  S.Clear;
  ReadStrings(Section, S);
end;

procedure TProfile.WriteStrings(const Section: string; S: TStrings);
var
  i: Integer;
begin
  WriteInteger(Section, 'Count', S.Count);
  for i := 0 to S.Count - 1 do
    WriteString(Section, 'S' + IntToStr(i), S[i]);
end;

procedure TProfile.WriteNewStrings(const Section: string; S: TStrings);
begin
  EraseSection(Section);
  WriteStrings(Section, S);
end;

procedure TProfile.WriteSectionValues(const Section: string; S: TStrings);
var
  i: Integer;
begin
  for i := 0 to S.Count - 1 do
    WriteString(Section, GetStrKey(S[i]), GetStrValue(S[i]));
end;

procedure TProfile.ReadFont(const Section: string; Font: TFont);
var
  s: string[63];
  n: Integer;
  fs: TFontStyles;
begin
  s := ReadString(Section,  'FontName', '');
  if s > '' then Font.Name := s;
  n := ReadInteger(Section, 'FontSize', 0);
  if n > 0 then Font.Size := n;
  s := Lowercase(ReadString(Section, 'FontStyle', ''));
  if s > '' then
  begin
    fs := [];
    if Pos('bold', s) > 0 then Include(fs, fsBold);
    if Pos('italic', s) > 0 then Include(fs, fsItalic);
    if Pos('underline', s) > 0 then Include(fs, fsUnderline);
    if Pos('strikeout', s) > 0 then Include(fs, fsStrikeOut);
    Font.Style := fs;
  end
  else Font.Style := [];
  s := ReadString(Section, 'FontColor', '');
  if s > '' then Font.Color := StringToColor(s);
end;

procedure TProfile.WriteFont(const Section: string; Font: TFont);
var
  s: string[63];
begin
  with Font do begin
    WriteString(Section, 'FontName', Name);
    WriteInteger(Section, 'FontSize', Size);
    s := '';
    if fsBold in Style then AppendStr(s, 'Bold ');
    if fsItalic in Style then AppendStr(s, 'Italic ');
    if fsUnderline in Style then AppendStr(s, 'Underline ');
    if fsStrikeOut in Style then AppendStr(s, 'Strikeout ');
    WriteString(Section, 'FontStyle', s);
    WriteString(Section, 'FontColor', ColorToString(Font.Color));
  end;
end;

procedure TProfile.ReadHeader(const Section: string; Header: THeader);
var
  i, w: Integer;
begin
  for i := 0 to ReadInteger(Section, 'SectionCount', 0) do
    with Header do
      if i < Sections.Count then
      begin
        w := ReadInteger(Section, 'Section' + IntToStr(i), -1);
        if w > -1 then SectionWidth[i] := w;
      end;
end;

procedure TProfile.WriteHeader(const Section: string; Header: THeader);
var
  i: Integer;
begin
  with Header do
  begin
    WriteInteger(Section, 'SectionCount', Sections.Count);
    for i := 0 to Sections.Count - 1 do
      WriteInteger(Section, 'Section' + IntToStr(i), SectionWidth[i]);
  end;
end;

end.

