 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * "gemdos.c" Version 1.7 Last delta 11:52:28 3/28/85      *
  *                                                         *
  * RESPONSIBLE ENGINEER:  G. M. Harding                    *
  *                                                         *
  * The revision history of this file is under the manage-  *
  * ment of SCCS on "drivax" (Pluto). Its associated s-file *
  * resides in /vis/harding/sccs/gdos, accessible to anyone *
  * in group "lang." Use the appropriate SCCS utilities for *
  * dates, versions, and comments. KERMIT is available on   *
  * Pluto, and may be used to download the file.            *
  *                                                         *
  * This file contains the system calls for basic I/O under *
  * GEMDOS.                                                 *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "portab.h"
#include "osif.h"

extern long trap();  /* System function access trap: RTA.S  */

#if GEMDOS

#include "osiferr.h"
#include "errno.h"

static int
   gdosmap(i)  /* Local function to translate error numbers */
REG long i;
{
   if (i < 0L)
      i *= (-1L);
   switch ( (int) i )
      {
      case  1:             /* Catch-all error number        */
         return (EINVAL);
      case  2:             /* Drive not ready               */
         return (ENODEV);
      case  3:             /* Unknown command               */
         return (ENOEXEC);
      case  4:             /* CRC error                     */
         return (EIO);
      case  5:             /* Bad request                   */
         return (EIO);
      case  6:             /* Seek error                    */
         return (EIO);
      case  7:             /* Unknown media                 */
         return (EIO);
      case  8:             /* Sector not found              */
         return (EIO);
      case  9:             /* No paper                      */
         return (EIO);
      case 10:             /* Write fault                   */
         return (EIO);
      case 11:             /* Read fault                    */
         return (EIO);
      case 12:             /* General mishap                */
         return (EIO);
      case 13:             /* Write protect                 */
         return (EROFS);
      case 14:             /* Media change                  */
         return (EIO);
      case 15:             /* Unknown device                */
         return (ENODEV);
      case 16:             /* Bad sectors on format         */
         return (EIO);
      case  0:             /* Shouldn't ever happen, but... */
      default:
         return (0);
      }
}

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #1 -- GEMDOS FUNCTION 1A -- SPECIFICATION V10, PAGE 42  *
  *                                                         *
  * This function performs the "setdta" function under GEM- *
  * DOS.                                                    *
  *                                                         *
  * USAGE:  VOID dos2setdta(buff)                           *
  *                                                         *
  * ARGUMENTS:                                              *
  *         BYTE *buff                                      *
  *                                                         *
  * RETURN VALUE:                                           *
  *         None                                            *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

VOID dos2setdta(buff)

BYTE *buff;     /* Pointer to DMA disk buffer in memory     */

{/* * * * * * * START OF  dos2setdta() FUNCTION * * * * * * */

   trap(SETTRA,buff);

}/* * * * * * *  END OF  dos2setdta() FUNCTION  * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #2 -- GEMDOS FUNCTION 2A -- SPECIFICATION V10, PAGE 43  *
  *                                                         *
  * This function performs the "get date" function under    *
  * GEMDOS.                                                 *
  *                                                         *
  * USAGE:  int dos2gdate(mon,day,yr)                       *
  *                                                         *
  * ARGUMENTS:                                              *
  *         int *mon                                        *
  *         int *day                                        *
  *         int *yr                                         *
  *                                                         *
  * RETURN VALUE:                                           *
  *         SUCCESS                                         *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2gdate(mon,day,yr)

int *mon, *day, *yr;    /* Pointers to ints to hold values  */

{/* * * * * * *  START OF dos2gdate() FUNCTION  * * * * * * */

   int k = (int)(trap(G_DATE) & 0xffffL);

   *day = k & 0x1f;
   *mon = (k >> 5) & 0xf;
   *yr  = ((k >> 9) & 0x7f) + 1980;

   return (SUCCESS);

}/* * * * * * * * END OF dos2gdate() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #3 -- GEMDOS FUNCTION 2C -- SPECIFICATION V10, PAGE 45  *
  *                                                         *
  * This function performs the "get time" function under    *
  * GEMDOS.                                                 *
  *                                                         *
  * USAGE:  int dos2gtime(hr,min,sec)                       *
  *                                                         *
  * ARGUMENTS:                                              *
  *         int *hr                                         *
  *         int *min                                        *
  *         int *sec                                        *
  *                                                         *
  * RETURN VALUE:                                           *
  *         SUCCESS                                         *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2gtime(hr,min,sec)

int *hr, *min, *sec;    /* Pointers to ints to hold values  */

{/* * * * * * *  START OF dos2gtime() FUNCTION  * * * * * * */

   int k = (int)(trap(G_TIME) & 0xffffL);

   *sec = (k & 0x1f) << 1;
   *min = (k >> 5) & 0x3f;
   *hr  = (k >> 11) & 0x1f;

   return (SUCCESS);

}/* * * * * * * * END OF dos2gtime() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #4 -- GEMDOS FUNCTION 30 -- SPECIFICATION V10, PAGE 48  *
  *                                                         *
  * This function performs the "get version" function under *
  * GEMDOS.                                                 *
  *                                                         *
  * USAGE:  int dos2version()                               *
  *                                                         *
  * ARGUMENTS:                                              *
  *         None                                            *
  *                                                         *
  * RETURN VALUE:                                           *
  *         16-bit integer with minor version in low byte,  *
  *         major version in high byte                      *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2version()

{/* * * * * * * START OF dos2version() FUNCTION * * * * * * */

   return ((int)(trap(GETVER) & 0xffffL));

}/* * * * * * *  END OF dos2version() FUNCTION  * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #5 -- GEMDOS FUNCTION 39 -- SPECIFICATION V10, PAGE 51  *
  *                                                         *
  * This function performs the "mkdir" function under GEM-  *
  * DOS.                                                    *
  *                                                         *
  * USAGE:  int dos2mkdir(dname)                            *
  *                                                         *
  * ARGUMENTS:                                              *
  *         char *dname                                     *
  *                                                         *
  * RETURN VALUE:                                           *
  *         SUCCESS or FAILURE                              *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2mkdir(dname)

char *dname;    /* Pointer to directory path name           */

{/* * * * * * *  START OF dos2mkdir() FUNCTION  * * * * * * */

   long k;

   if ((k = trap(MKDIR,dname)) == 0L)
      return (SUCCESS);
   else
      RETERR(FAILURE,(gdosmap(k)))

}/* * * * * * * * END OF dos2mkdir() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #6 -- GEMDOS FUNCTION 3A -- SPECIFICATION V10, PAGE 52  *
  *                                                         *
  * This function performs the "rmdir" function under GEM-  *
  * DOS.                                                    *
  *                                                         *
  * USAGE:  int dos2rmdir(dname)                            *
  *                                                         *
  * ARGUMENTS:                                              *
  *         char *dname                                     *
  *                                                         *
  * RETURN VALUE:                                           *
  *         SUCCESS or FAILURE                              *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2rmdir(dname)

char *dname;    /* Pointer to directory path name           */

{/* * * * * * *  START OF dos2rmdir() FUNCTION  * * * * * * */

   long k;

   if ((k = trap(RMDIR,dname)) == 0L)
      return (SUCCESS);
   else
      RETERR(FAILURE,(gdosmap(k)))

}/* * * * * * * * END OF dos2rmdir() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #7 -- GEMDOS FUNCTION 3B -- SPECIFICATION V10, PAGE 53  *
  *                                                         *
  * This function performs the "chdir" function under GEM-  *
  * DOS.                                                    *
  *                                                         *
  * USAGE:  int dos2chdir(dname)                            *
  *                                                         *
  * ARGUMENTS:                                              *
  *         char *dname                                     *
  *                                                         *
  * RETURN VALUE:                                           *
  *         SUCCESS or FAILURE                              *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2chdir(dname)

char *dname;    /* Pointer to directory path name           */

{/* * * * * * *  START OF dos2chdir() FUNCTION  * * * * * * */

   long k;

   if ((k = trap(CHDIR,dname)) == 0L)
      return (SUCCESS);
   else
      RETERR(FAILURE,(gdosmap(k)))

}/* * * * * * * * END OF dos2chdir() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #8 -- GEMDOS FUNCTION 3C -- SPECIFICATION V10, PAGE 54  *
  *                                                         *
  * This function performs file creates under GEMDOS.       *
  *                                                         *
  * USAGE:  int dos2creat(fname, attrib)                    *
  *                                                         *
  * ARGUMENTS:                                              *
  *         char *fname                                     *
  *         int attrib                                      *
  *                                                         *
  * RETURN VALUE:                                           *
  *         File descriptor, or -1 if unsuccessful.         *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2creat(fname,attrib)

char *fname;    /* Pointer to null-terminated pathname      */

int attrib;     /* Logical AND of the following:            */
                /*      01H => Read-only                    */
                /*      02H => Invisible                    */
                /*      04H => System                       */
                /*      08H => Contains volume label        */

{/* * * * * * *  START OF dos2creat() FUNCTION  * * * * * * */

   long k;

   if ((k = trap(CREATE,fname,attrib)) < 0L)
      RETERR(FAILURE,(gdosmap(k)))
   else
      return ((int) k );

}/* * * * * * * * END OF dos2creat() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #9 -- GEMDOS FUNCTION 3D -- SPECIFICATION V10, PAGE 56  *
  *                                                         *
  * This function performs file opens under GEMDOS.         *
  *                                                         *
  * USAGE:  int dos2open(fname, mode)                       *
  *                                                         *
  * ARGUMENTS:                                              *
  *         char *fname                                     *
  *         int mode                                        *
  *                                                         *
  * RETURN VALUE:                                           *
  *         File descriptor, or -1 if failure.              *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2open(fname,mode)

char *fname;    /* Pointer to null-terminated pathname      */

int mode;       /* 0 => R    1 => W    2 => R/W             */

{/* * * * * * *  START OF  dos2open() FUNCTION  * * * * * * */

   long k;

   if ((k = trap(OPEN,fname,mode)) < 0L)
      RETERR(FAILURE,(gdosmap(k)))
   else
      return ((int) k );

}/* * * * * * * * END OF  dos2open() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #10 -- GEMDOS FUNCTION 3E -- SPECIFICATION V10, PAGE 57 *
  *                                                         *
  * This function performs file closes under GEMDOS.        *
  *                                                         *
  * USAGE:  int dos2close(fd)                               *
  *                                                         *
  * ARGUMENTS:                                              *
  *         int fd                                          *
  *                                                         *
  * RETURN VALUE:                                           *
  *         SUCCESS or FAILURE                              *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2close(fd)

int fd;         /* File descriptor from dos2open()          */

{/* * * * * * *  START OF dos2close() FUNCTION  * * * * * * */

   long k;

   if ((k = trap(CLOSE,fd)) == 0L)
      return (SUCCESS);
   else
      RETERR(FAILURE,(gdosmap(k)))

}/* * * * * * * * END OF dos2close() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #11 -- GEMDOS FUNCTION 3F -- SPECIFICATION V10, PAGE 58 *
  *                                                         *
  * This function performs the "read" function under GEM-   *
  * DOS. It is the caller's responsibility to insure that   *
  * the receiving buffer is large enough to hold the number *
  * of bytes requested.                                     *
  *                                                         *
  * USAGE:  int dos2read(fd,buffer,bytes)                   *
  *                                                         *
  * ARGUMENTS:                                              *
  *         int fd                                          *
  *         char *buffer                                    *
  *         unsigned bytes                                  *
  *                                                         *
  * RETURN VALUE:                                           *
  *         Number of bytes actually read, 0 on EOF, or     *
  *         FAILURE                                         *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2read(fd,buffer,bytes)

int fd;         /* File descriptor of file to read from     */

char *buffer;   /* Buffer to hold data read from the file   */

unsigned bytes; /* Number of bytes to be read from the file */

{/* * * * * * *  START OF  dos2read() FUNCTION  * * * * * * */

   long k;

   if ((k = trap(B_READ, fd, (long) bytes, buffer)) < 0L)
      RETERR(FAILURE,(gdosmap(k)))
   else
      return ((int) k );

}/* * * * * * * * END OF  dos2read() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #12 -- GEMDOS FUNCTION 40 -- SPECIFICATION V10, PAGE 60 *
  *                                                         *
  * This function performs the "write" function under GEM-  *
  * DOS.                                                    *
  *                                                         *
  * USAGE:  int dos2write(fd,buffer,bytes)                  *
  *                                                         *
  * ARGUMENTS:                                              *
  *         int fd                                          *
  *         char *buffer                                    *
  *         unsigned bytes                                  *
  *                                                         *
  * RETURN VALUE:                                           *
  *         Number of bytes actually written, or FAILURE    *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2write(fd,buffer,bytes)

int fd;         /* File descriptor of file to write to      */

char *buffer;   /* Buffer holding data to write to the file */

unsigned bytes; /* Number of bytes to write to the file     */

{/* * * * * * *  START OF dos2write() FUNCTION  * * * * * * */

   long k;

   if ((k = trap(B_WRITE, fd, (long) bytes, buffer)) < 0L)
      RETERR(FAILURE,(gdosmap(k)))
   else
      return ((int) k );

}/* * * * * * * * END OF dos2write() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #13 -- GEMDOS FUNCTION 41 -- SPECIFICATION V10, PAGE 62 *
  *                                                         *
  * This function performs the "era" function under GEMDOS. *
  *                                                         *
  * USAGE:  int dos2unlink(name)                            *
  *                                                         *
  * ARGUMENTS:                                              *
  *         char *name                                      *
  *                                                         *
  * RETURN VALUE:                                           *
  *         SUCCESS or FAILURE                              *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2unlink(name)

char *name;     /* Pointer to path naming file to be killed */

{/* * * * * * * START OF  dos2unlink() FUNCTION * * * * * * */

   long k;

   if ((k = trap(DELETE,name)) == 0L)
      return (SUCCESS);
   else
      RETERR(FAILURE,(gdosmap(k)))

}/* * * * * * *  END OF  dos2unlink() FUNCTION  * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #14 -- GEMDOS FUNCTION 42 -- SPECIFICATION V10, PAGE 63 *
  *                                                         *
  * This function performs the "lseek" function under GEM-  *
  * DOS.                                                    *
  *                                                         *
  * USAGE:  long dos2lseek(fd,offset,sense)                 *
  *                                                         *
  * ARGUMENTS:                                              *
  *         int fd                                          *
  *         long offset                                     *
  *         int sense                                       *
  *                                                         *
  * RETURN VALUE:                                           *
  *         Absolute file offset, or -1 if failure          *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

long dos2lseek(fd,offset,sense)

int fd;         /* Valid file descriptor from dos2open()    */

long offset;    /* Offset into the file                     */

int sense;      /* Meaning of offset: 0 => From beginning   */
                /*                    1 => From curr. posn. */
                /*                    2 => From end of file */

{/* * * * * * *  START OF dos2lseek() FUNCTION  * * * * * * */

   long k;

   if ((k = trap(LSEEK,offset,fd,sense)) < 0L)
      RETERR(FAILURE,(gdosmap(k)))
   else
      return (k);

}/* * * * * * * * END OF dos2lseek() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #15 -- GEMDOS FUNCTION 45 -- SPECIFICATION V10, PAGE 67 *
  *                                                         *
  * This function performs the "dup" function under GEMDOS. *
  *                                                         *
  * USAGE:  int dos2dup(fd)                                 *
  *                                                         *
  * ARGUMENTS:                                              *
  *         int fd                                          *
  *                                                         *
  * RETURN VALUE:                                           *
  *         FAILURE, or the duplicate fd                    *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2dup(fd)

int fd;         /* File descriptor of file to dup           */

{/* * * * * * * * START OF dos2dup() FUNCTION * * * * * * * */

   long k;

   if ((k = trap(DUP,fd)) >= 0L)
      return ((int) k );
   else
      RETERR(FAILURE,(gdosmap(k)))

}/* * * * * * * *  END OF dos2dup() FUNCTION  * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #16 -- GEMDOS FUNCTION 4B -- SPECIFICATION V10, PAGE 73 *
  *                                                         *
  * This function simulates "dos2exec". FOR NOW, it is only *
  * a stub which always exits to the OS. If dos2exec() is   *
  * successful, it does not return, because it overlays its *
  * calling process with a new process. If it does return,  *
  * it returns the value -1, and sets errno to the value of *
  * the cause of the failure.                               *
  *                                                         *
  * USAGE:  int dos2exec(pname,parms,func)                  *
  *                                                         *
  * ARGUMENTS:                                              *
  *         char *pname                                     *
  *         char *parms                                     *
  *         int func                                        *
  *                                                         *
  * RETURN VALUE:                                           *
  *         FAILURE, if it returns at all                   *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2exec(pname,parms,func)

char *pname;    /* Pointer to complete pathname to execute  */

char *parms;    /* Pointer to parameter block               */

int func;       /* Function: 0 => Load and execute program  */
                /*           1 => Load but do not execute   */

{/* * * * * * *  START OF  dos2exec() FUNCTION  * * * * * * */

   return ((int)trap(EXITWSC,-1));           /* FOR NOW     */

/*
  +---------------------------------------------
  |                  FOR LATER
  | Add pointers to command tail and environment
  | string.
  +---------------------------------------------

   long k = trap(EXEC,func * 3,pname, (char *) 0, (char *) 0);

   if (k < 0L)
      RETERR(FAILURE,ENOENT)
   else
      return ((int) k );

*/

}/* * * * * * * * END OF  dos2exec() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #17 -- GEMDOS FUNCTION 4C -- SPECIFICATION V10, PAGE 75 *
  *                                                         *
  * This function performs the "exit to OS" function under  *
  * GEMDOS, passing an exit status argument. Even though    *
  * the function is defined as being of type integer, it is *
  * effectively a void function, and should be so called.   *
  *                                                         *
  * USAGE:  VOID dos2exit(code)                             *
  *                                                         *
  * ARGUMENTS:                                              *
  *         int code                                        *
  *                                                         *
  * RETURN VALUE:                                           *
  *         None (function never returns)                   *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2exit(code)

int code;       /* Exit status to be returned to OS         */

{/* * * * * * *  START OF  dos2exit() FUNCTION  * * * * * * */

   return ((int)trap(EXITWSC,code));

}/* * * * * * * * END OF  dos2exit() FUNCTION * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #18 -- GEMDOS FUNCTION 4E -- SPECIFICATION V10, PAGE 77 *
  *                                                         *
  * This function performs the "sfirst" function under GEM- *
  * DOS.                                                    *
  *                                                         *
  * USAGE:  WORD dos2f_file(fname,attr)                     *
  *                                                         *
  * ARGUMENTS:                                              *
  *         BYTE *fname                                     *
  *         WORD attr                                       *
  *                                                         *
  * RETURN VALUE:                                           *
  *         SUCCESS or FAILURE                              *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

WORD dos2f_file(fname,attr)

BYTE *fname;    /* Pointer to name, possibly with wildcards */

WORD attr;      /* Logical AND of the following:            */
                /*      01H => Read-only                    */
                /*      02H => Invisible                    */
                /*      04H => System                       */
                /*      08H => Contains volume label        */
                /*      10H => Subdirectory                 */
                /*      20H => File has been written to     */

{/* * * * * * * START OF  dos2f_file() FUNCTION * * * * * * */

   if (trap(SFIRST, fname, (int) attr) & 0xffffL)
      return (SUCCESS);
   else
      return (FAILURE);

}/* * * * * * *  END OF  dos2f_file() FUNCTION  * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #19 -- GEMDOS FUNCTION 4F -- SPECIFICATION V10, PAGE 79 *
  *                                                         *
  * This function performs the "snext" function under GEM-  *
  * DOS.                                                    *
  *                                                         *
  * USAGE:  WORD dos2f_next()                               *
  *                                                         *
  * ARGUMENTS:                                              *
  *         None                                            *
  *                                                         *
  * RETURN VALUE:                                           *
  *         SUCCESS or FAILURE                              *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

WORD dos2f_next()

{/* * * * * * * START OF  dos2f_next() FUNCTION * * * * * * */

   if (trap(SNEXT) & 0xffffL)
      return (SUCCESS);
   else
      return (FAILURE);

}/* * * * * * *  END OF  dos2f_next() FUNCTION  * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #20 -- GEMDOS FUNCTION 56 -- SPECIFICATION V10, PAGE 80 *
  *                                                         *
  * This function performs the "rename" function under GEM- *
  * DOS.                                                    *
  *                                                         *
  * USAGE:  int dos2rename(from,to)                         *
  *                                                         *
  * ARGUMENTS:                                              *
  *         char *from                                      *
  *         char *to                                        *
  *                                                         *
  * RETURN VALUE:                                           *
  *         SUCCESS or FAILURE                              *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2rename(from,to)

char *from, *to;   /* Pointers to old and new file names    */

{/* * * * * * * START OF  dos2rename() FUNCTION * * * * * * */

   long k;

   if ((k = trap(RENAME,from,to)) == 0L)
      return (SUCCESS);
   else
      RETERR(FAILURE,(gdosmap(k)))

}/* * * * * * *  END OF  dos2rename() FUNCTION  * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  *                                                         *
  * #21 -- DUMMY -- NO CORRESPONDING GEMDOS FUNCTION        *
  *                                                         *
  * This function simulates "ioctl" for GEMDOS. Actually,   *
  * it is a dummy function which always returns FAILURE, in *
  * order to force a call to dos2read() rather than ttyin() *
  * (the tty-specific function under PC-DOS).               *
  *                                                         *
  * USAGE:  int dos2ioctl(fd,drive,buf,cnt,req)             *
  *                                                         *
  * ARGUMENTS:                                              *
  *         int fd                                          *
  *         int drive                                       *
  *         char *buf                                       *
  *         int cnt                                         *
  *         int req                                         *
  *                                                         *
  * RETURN VALUE:                                           *
  *         Always returns FAILURE                          *
  *                                                         *
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int dos2ioctl(fd,drive,buf,cnt,req)

int fd;         /* File descriptor of file to check         */

int drive;      /* 0 = default, 1 = A, 2 = B, etc.          */

char *buf;      /* Data or buffer                           */

int cnt;        /* Number of bytes to read/write            */

int req;        /* Request:  0 => Get device information    */
                /*           1 => Set device information    */
                /*           2 => Read to buf from dev CCB  */
                /*           3 => Write from buf to dev CCB */
                /*           4 => Read to buf from drive #  */
                /*           5 => Write from buf to drive # */
                /*           6 => Get input status          */
                /*           7 => Set input status          */

{/* * * * * * *  START OF dos2ioctl() FUNCTION  * * * * * * */

   if (req > 0)
      RETERR(FAILURE,EINVAL)
   else
      return (FAILURE);                      /* FOR NOW     */

}/* * * * * * * * END OF dos2ioctl() FUNCTION * * * * * * * */

#endif




 /* * * * * * * * * * * END OF gemdos.c * * * * * * * * * * */

             